from collections.abc import AsyncGenerator
from typing import Any

from sqlalchemy import Column, select
from sqlalchemy.ext.asyncio import AsyncSession, create_async_engine
from sqlalchemy.orm import as_declarative, declared_attr, sessionmaker

from config import DATABASE_URL


@as_declarative()
class Base:
    id: Any
    __name__: str

    @declared_attr
    def __tablename__(cls) -> str:
        return cls.__name__

    @declared_attr
    def __all_fields__(cls) -> list[str]:
        return cls.__table__.columns.keys()

    @declared_attr
    def __columns_items__(cls) -> dict[str, Column]:
        """
        Пример:
            ```python
                select(
                    Table.field1,
                    array_agg(
                        func.jsonb_build_object(
                            *chain(*Table.__columns_items__)
                        )
                    )
                ).group_by(Table.field1)
            ```
        """
        return cls.__table__.columns.items()

    def __repr__(self) -> str:
        items = [
            f'{field}: {getattr(self, field)}'
            for field in self.__table__.columns
        ]
        return f'{self.__tablename__}({", ".join(items)})'

    @classmethod
    async def get_or_create(
        cls,
        defaults: dict | None = None,
        **kwargs,
    ):
        async with async_session_maker() as session, session.begin():
            result = await session.execute(
                select(cls).filter_by(**kwargs),
            )
            instance = result.scalar_one_or_none()

            if instance:
                return instance, False

            params = {
                key: value for key, value in kwargs.items()
                if key in cls.__all_fields__
            }
            if defaults:
                params.update(defaults)

            instance = cls(**params)
            session.add(instance=instance)

            return instance, True


engine = create_async_engine(DATABASE_URL)
async_session_maker = sessionmaker(
    engine,
    class_=AsyncSession,
    expire_on_commit=False,
)


async def get_async_session() -> AsyncGenerator[AsyncSession, None]:
    async with async_session_maker() as session:
        yield session
