from functools import wraps

from database import async_session_maker
from services.state.state import State
from services.state.storage import RedisStorage


DEFAULT_STORAGE = RedisStorage


def _init_state() -> State:
    """
    Инициализация коннекта с БД
    :return: коннект БД
    """
    return State(DEFAULT_STORAGE())


def database_async_client_decorator(func):
    """
    Для асинхронных функций
    Декоратор для инициализации и автоматического закрытия коннекта с БД
    """
    async def wrapper(*args, **kwargs):
        yield await func(*args, state=_init_state(), **kwargs)
    return wrapper


def database_async_client(func):
    async def wrapper(*args, **kwargs):
        await func(*args, state=_init_state(), **kwargs)
    return wrapper


class WithSession:
    def __call__(self, func):
        @wraps(func)
        async def wrapper(*args, **kwargs):
            async with async_session_maker() as session, session.begin():
                await func(*args, **kwargs, session=session)
        return wrapper
