import logging
import time
from datetime import datetime, timedelta
from itertools import islice

import backoff
import jwt
import requests
from requests import RequestException, Response

import config
from custom_types import ComputedItem, RequestDataItem
from services.datasphere.datasphere import BaseDataSphere


logger = logging.getLogger(__name__)


class YandexCloudToken:
    """Класс получения токена для Яндекс клауд"""
    __iam_token: str = None
    __token_generated_at: datetime.utcnow = None
    iam_token_create_url = 'https://iam.api.cloud.yandex.net/iam/v1/tokens'

    @classmethod
    def _generate_yandex_jwt(cls) -> str:
        """Генерация JWT токена"""
        now = int(time.time())
        payload = {
            'aud': cls.iam_token_create_url,
            'iss': config.ACAD_PERF_YANDEX_SERVICE_ACCOUNT_ID,
            'iat': now,
            'exp': now + 60,
        }
        private_key = config.ACAD_PERF_YANDEX_SECRET_KEY
        return jwt.encode(
            payload,
            private_key,
            algorithm='PS256',
            headers={'kid': config.ACAD_PERF_YANDEX_KEY_ID},
        )

    @classmethod
    def _generate_iam_token(cls) -> None:
        """Обмен JWT токена на IAM токен"""
        yandex_jwt = cls._generate_yandex_jwt()
        response = requests.post(
            cls.iam_token_create_url,
            json={'jwt': yandex_jwt},
            headers={'Content-Type': 'application/json'},
        )
        if not response.ok:
            raise RequestException(response.text)
        response_data = response.json()
        cls.__iam_token = response_data['iamToken']
        cls.__token_generated_at = datetime.now()

    @classmethod
    def get_iam_token(cls) -> str:
        """Получение актуального токена"""
        if (
            cls.__iam_token and
            cls.__token_generated_at and
            cls.__token_generated_at > datetime.now() - timedelta(hours=config.ACAD_PERF_YANDEX_IAM_LIVE_TIME_HOURS)
        ):
            return cls.__iam_token
        cls._generate_iam_token()
        return cls.__iam_token


class YandexDataSphere(BaseDataSphere):
    """Класс работающий с Яндекс датасферой"""
    yandex_cloud: YandexCloudToken = None
    datasphere_url = 'https://datasphere.api.cloud.yandex.net/datasphere/v1/aliases/'

    def __init__(self):
        self.yandex_cloud = YandexCloudToken()

    def _prepare_request_url(self) -> str:
        """Генерация url для запроса"""
        return f'{self.datasphere_url}{config.ACAD_PERF_DATASPHERE_ALIAS}:execute'

    def _prepare_request_headers(self) -> dict:
        """Формирование заголовков запроса"""
        iam_token = self.yandex_cloud.get_iam_token()
        return {
            'Authorization': f'Bearer {iam_token}',
            'Content-Type': 'application/json',
        }

    def _convert_request_data_items_from_data(self, data: dict[str, RequestDataItem]) -> dict[str, list]:
        return {
            key: list(value) for key, value in data.items()
        }

    def _prepare_request_body(self, data: dict[str, RequestDataItem]) -> dict:
        """Формирование тела запроса"""
        return {
            'folder_id': config.ACAD_PERF_DATASPHERE_FOLDER_ID,
            'input': {
                'input_data': self._convert_request_data_items_from_data(data),
            },
        }

    def _split_by_chunks(self, data):
        """Разбиение данных на пачки"""
        it = iter(data)
        for _i in range(0, len(data), config.ACAD_PERF_DATASPHERE_CHUNK_SIZE):
            yield {k: data[k] for k in islice(it, config.ACAD_PERF_DATASPHERE_CHUNK_SIZE)}

    @staticmethod
    def _response_validation(request_data: dict[str, RequestDataItem], response: Response):
        # Если что-то "не ОК", то логируем ошибку
        if not response.ok:
            logger.error('Send to datasphere error: %s', response.text)
            raise RequestException(response.text)

        # Если отправлен не верный запрос, то нам вернутся не корректные данные
        # Проверяем это, сверяя ключи
        response_data = response.json()['output']['output_data']
        for key in response_data:
            if not request_data.get(key):
                msg = f'Invalid response data\nRequest data - {request_data}\nResponse data - {response_data}\n'
                raise RequestException(msg)

        info_message = f'Data (chunk_max: {config.ACAD_PERF_DATASPHERE_CHUNK_SIZE}) sent to datasphere'
        logger.info(msg=info_message)

    @backoff.on_exception(
        backoff.expo, requests.exceptions.RequestException, max_time=10, max_tries=None, jitter=backoff.full_jitter)
    def _send_chunk_to_datasphere(self, data: dict[str, RequestDataItem]) -> dict[str, ComputedItem]:
        """Отправка отдельной пачки в датасферу"""
        response = requests.post(
            url=self._prepare_request_url(),
            json=self._prepare_request_body(data),
            headers=self._prepare_request_headers(),
        )
        self._response_validation(data, response)
        response_data = response.json()
        return response_data['output']['output_data']

    def send(self, request_data: dict[str, RequestDataItem]) -> dict[str, ComputedItem]:
        """Отправка данных в датасферу"""
        chunks = self._split_by_chunks(request_data)
        output_data = {}
        for chunk in chunks:
            output_data.update(self._send_chunk_to_datasphere(chunk))
        return output_data
