import abc
from collections.abc import Iterable

from config import ACAD_PERF_KAFKA_TOPIC, ACAD_PERF_SERVICE_ID
from custom_types import RawItem


CACHE_PREFIX_SERVICE_NAME = f'{ACAD_PERF_KAFKA_TOPIC}/'
CACHE_PREFIX_SERVICE_UPDATE = f'{CACHE_PREFIX_SERVICE_NAME}{ACAD_PERF_SERVICE_ID}/'
CACHE_PREFIX_KAFKA_RAW = f'{CACHE_PREFIX_SERVICE_UPDATE}raw/'
CACHE_PREFIX_KAFKA_OFFSET = f'{CACHE_PREFIX_SERVICE_NAME}partition_offset/'
CACHE_PREFIX_DATASPHERE_PENDING = f'{CACHE_PREFIX_SERVICE_UPDATE}pending/'


class StateStorage(abc.ABC):
    """Интерфейс работы с состоянием"""

    @abc.abstractmethod
    def set_raw_item(self, key: str, data: RawItem) -> None:
        """Сохранение сырых данных"""

    @abc.abstractmethod
    def add_pending_item(self, data: RawItem) -> None:
        """Сохранение ожидающих данных"""

    @abc.abstractmethod
    def replace_pending_data(self, data: list[RawItem]) -> None:
        """Сохранение ожидающих данных"""

    @abc.abstractmethod
    def set_kafka_offset(self, partition_id: int, offset: int) -> None:
        """Сохранение смещения в кафка"""

    @abc.abstractmethod
    def get_all_raw(self) -> dict[str, RawItem]:
        """Получить все сырые записи"""

    @abc.abstractmethod
    def get_item_raw(self, key: str) -> RawItem:
        """Получить одну сырую запись по ключу"""

    @abc.abstractmethod
    def get_all_pending(self) -> list[RawItem]:
        """Получить все записи на ожидании"""

    @abc.abstractmethod
    def get_kafka_offset(self, partition_id: int) -> int:
        """Получение смещения в кафка"""

    @abc.abstractmethod
    def clean_raw(self, keys: Iterable[str]) -> None:
        """Очистка сырых данных"""


class State:
    def __init__(self, storage: StateStorage):
        self.storage = storage

    def _pending_data_processing(self):
        """Обработка отложенных данных"""
        new_pending_data = []
        pending_data = self.storage.get_all_pending()
        raw_data = self.storage.get_all_raw()
        for pending_item in pending_data:
            pending_hash = pending_item['hash_value']
            if not raw_data.get(pending_hash):
                raw_data[pending_hash] = pending_item
                self.storage.set_raw_item(pending_hash, pending_item)
                continue
            new_pending_data.append(pending_item)
        self.storage.replace_pending_data(new_pending_data)

    def clean_raw(self, raw_keys: list[str]) -> None:
        """
        Метод сохранения состояния после обработки данных
        :param raw_keys: Список ключей для удаления
        """
        self.storage.clean_raw(raw_keys)

    def get_all_raw(self) -> dict[str, RawItem]:
        """Получение всей сырой информации"""
        self._pending_data_processing()
        return self.storage.get_all_raw()

    def get_kafka_offset(self, partition_id: int) -> int:
        """Получение данных о сдвиге в кафка"""
        return self.storage.get_kafka_offset(partition_id)

    def add_raw_item(self, data: RawItem, partition_id: int, offset: int) -> None:
        """Добавление элемента не обработанной информации и обновление сдвига"""
        key = data['hash_value']
        if self.storage.get_item_raw(key):
            self.storage.add_pending_item(data)
        else:
            self.storage.set_raw_item(key, data)
        self.storage.set_kafka_offset(partition_id, offset)
