import json
from collections.abc import Iterable

import etcd3
from redis import Redis

from config import (
    ACAD_PERF_REDIS_DB,
    ACAD_PERF_STORAGE_HOST,
    ACAD_PERF_STORAGE_PASSWORD,
    ACAD_PERF_STORAGE_PATH_TO_CA_CERT,
    ACAD_PERF_STORAGE_PORT,
    ACAD_PERF_STORAGE_USER,
)
from custom_types import RawItem
from services.state.state import (
    CACHE_PREFIX_DATASPHERE_PENDING,
    CACHE_PREFIX_KAFKA_OFFSET,
    CACHE_PREFIX_KAFKA_RAW,
    StateStorage,
)


class EtcdStorage(StateStorage):
    """
    Реализация хранилища в etcd
    """
    __db_connect = None

    @property
    def connect(self) -> etcd3.Etcd3Client:
        return self.__db_connect or self._init_db_connect()

    def _init_db_connect(self) -> etcd3.Etcd3Client:
        if ACAD_PERF_STORAGE_PATH_TO_CA_CERT:
            self.__db_connect = etcd3.client(
                ca_cert=ACAD_PERF_STORAGE_PATH_TO_CA_CERT,
                user=ACAD_PERF_STORAGE_USER,
                password=ACAD_PERF_STORAGE_PASSWORD,
                host=ACAD_PERF_STORAGE_HOST,
                port=ACAD_PERF_STORAGE_PORT,
            )
        else:
            self.__db_connect = etcd3.client(
                host=ACAD_PERF_STORAGE_HOST,
                port=ACAD_PERF_STORAGE_PORT,
            )
        return self.__db_connect

    def set_raw_item(self, key: str, data: RawItem) -> None:
        """Сохранение сырых данных"""
        self.connect.put(f'{CACHE_PREFIX_KAFKA_RAW}{key}', json.dumps(data))

    def set_kafka_offset(self, partition_id: int, offset: int) -> None:
        """Сохранение смещения в кафка"""
        self.connect.put(f'{CACHE_PREFIX_KAFKA_OFFSET}{partition_id}', str(offset))

    def add_pending_item(self, data: RawItem) -> None:
        """Сохранение ожидающих данных"""
        pending = self.get_all_pending()
        pending.append(data)
        self.replace_pending_data(pending)

    def replace_pending_data(self, data: list[RawItem]) -> None:
        """Сохранение ожидающих данных"""
        self.connect.put(CACHE_PREFIX_DATASPHERE_PENDING, json.dumps(data))

    def get_all_pending(self) -> list[RawItem]:
        """Получить все записи на ожидании"""
        data, _meta = self.connect.get(CACHE_PREFIX_DATASPHERE_PENDING)
        return json.loads(data) if data else []

    def get_all_raw(self) -> dict[str, RawItem]:
        """Получить все сырые записи"""
        data = {}
        for value, _meta in self.connect.get_prefix(CACHE_PREFIX_KAFKA_RAW):
            dict_value: RawItem = json.loads(value)
            data[dict_value['hash_value']] = dict_value
        return data

    def get_item_raw(self, key: str) -> RawItem:
        """Получить одну сырую запись по ключу"""
        data, _ = self.connect.get(f'{CACHE_PREFIX_KAFKA_RAW}{key}')
        return json.loads(data) if data else {}

    def get_kafka_offset(self, partition_id: int) -> int:
        """Получение смещения в кафка"""
        offset, _ = self.connect.get(f'{CACHE_PREFIX_KAFKA_OFFSET}{partition_id}')
        return int(offset) if offset else 0

    def clean_raw(self, keys: Iterable[str]) -> None:
        """Очистка сырых данных"""
        for key in keys:
            self.connect.delete(key=f'{CACHE_PREFIX_KAFKA_RAW}{key}')


class RedisStorage(StateStorage):
    __db_connect = None

    @property
    def connect(self) -> Redis:
        return self.__db_connect or self._init_db_connect()

    def _init_db_connect(self) -> Redis:
        if ACAD_PERF_STORAGE_PASSWORD:
            self.__db_connect = Redis(
                host=ACAD_PERF_STORAGE_HOST,
                port=ACAD_PERF_STORAGE_PORT,
                db=ACAD_PERF_REDIS_DB,
                password=ACAD_PERF_STORAGE_PASSWORD,
            )
        else:
            self.__db_connect = Redis(
                host=ACAD_PERF_STORAGE_HOST,
                port=ACAD_PERF_STORAGE_PORT,
                db=ACAD_PERF_REDIS_DB,
            )
        return self.__db_connect

    def set_raw_item(self, key: str, data: RawItem) -> None:
        """Сохранение сырых данных"""
        self.connect.set(f'{CACHE_PREFIX_KAFKA_RAW}{key}', json.dumps(data))

    def set_kafka_offset(self, partition_id: int, offset: int) -> None:
        """Сохранение смещения в кафка"""
        self.connect.set(f'{CACHE_PREFIX_KAFKA_OFFSET}{partition_id}', str(offset))

    def add_pending_item(self, data: RawItem) -> None:
        """Сохранение ожидающих данных"""
        pending = self.get_all_pending()
        pending.append(data)
        self.replace_pending_data(pending)

    def replace_pending_data(self, data: list[RawItem]) -> None:
        """Сохранение ожидающих данных"""
        self.connect.set(CACHE_PREFIX_DATASPHERE_PENDING, json.dumps(data))

    def get_all_pending(self) -> list[RawItem]:
        """Получить все записи на ожидании"""
        data = self.connect.get(CACHE_PREFIX_DATASPHERE_PENDING)
        return json.loads(data) if data else []

    def get_all_raw(self) -> dict[str, RawItem]:
        """Получить все сырые записи"""
        data = {}
        for keys in self.connect.keys(f'{CACHE_PREFIX_KAFKA_RAW}*'):
            value = self.connect.get(keys)
            dict_value: RawItem = json.loads(value)
            data[dict_value['hash_value']] = dict_value
        return data

    def get_item_raw(self, key: str) -> RawItem:
        """Получить одну сырую запись по ключу"""
        data = self.connect.get(f'{CACHE_PREFIX_KAFKA_RAW}{key}')
        return json.loads(data) if data else {}

    def get_kafka_offset(self, partition_id: int) -> int:
        """Получение смещения в кафка"""
        offset = self.connect.get(f'{CACHE_PREFIX_KAFKA_OFFSET}{partition_id}')
        return int(offset) if offset else 0

    def clean_raw(self, keys: Iterable[str]) -> None:
        """Очистка сырых данных"""
        for key in keys:
            self.connect.delete(f'{CACHE_PREFIX_KAFKA_RAW}{key}')
