"""
    Файл, содержащий определение и настройку Faust Streams.
    В этом файле можно создать несколько потоков данных и определить их схемы.
"""
import logging
from logging.config import dictConfig

import faust
from etcd3.exceptions import ConnectionFailedError
from faust.types import TP
from faust.web import View
from redis.exceptions import ConnectionError

from config import (
    ACAD_PERF_KAFKA_PARTITION_COUNT,
    ACAD_PERF_KAFKA_PASSWORD,
    ACAD_PERF_KAFKA_SASL_ENABLED,
    ACAD_PERF_KAFKA_SASL_MECHANISM,
    ACAD_PERF_KAFKA_SERVER_PROCESSED,
    ACAD_PERF_KAFKA_TOPIC,
    ACAD_PERF_KAFKA_USERNAME,
    ACAD_PERF_SEND_DATASPHERE_INTERVAL,
    LOGGING_CONFIG,
)
from custom_types import StateLivenessProbe
from decorators import database_async_client, database_async_client_decorator
from proto.schemas import TaskDoneEvent
from services.academic_performance import DataHandler, KafkaReader
from services.state.state import State


logger = logging.getLogger(__name__)

if ACAD_PERF_KAFKA_SASL_ENABLED:
    logger.info('Configure broker with SASL')
    print('Configure broker with SASL')
    broker_credentials = faust.SASLCredentials(
        username=ACAD_PERF_KAFKA_USERNAME,
        password=ACAD_PERF_KAFKA_PASSWORD,
        mechanism=ACAD_PERF_KAFKA_SASL_MECHANISM,
    )
else:
    logger.info('Configure broker without SASL')
    print('Configure broker without SASL')
    broker_credentials = None

app = faust.App(
    ACAD_PERF_KAFKA_TOPIC,
    broker=ACAD_PERF_KAFKA_SERVER_PROCESSED,
    broker_credentials=broker_credentials,
    logging_config=dictConfig(LOGGING_CONFIG),
)

performance_topic = app.topic(
    ACAD_PERF_KAFKA_TOPIC,
    schema=TaskDoneEvent,
    partitions=ACAD_PERF_KAFKA_PARTITION_COUNT,
)


@app.task()
@database_async_client
async def start_offset(*args, state: State):
    try:
        topic_partitions: set[TP] = {
            topic_partition
            for topic_partition in app.consumer.assignment()
            if topic_partition.topic == ACAD_PERF_KAFKA_TOPIC
        }
        for topic_partition in topic_partitions:
            offset = state.get_kafka_offset(topic_partition.partition)
            if offset:
                await app.consumer.seek(topic_partition, int(offset))
                message_offset_exist = f'{topic_partition}. Offset: {offset}'
                logger.info(msg=message_offset_exist)
            message_offset_not_exist = f'{topic_partition}. Offset: не найден в Local Storage'
            logger.info(msg=message_offset_not_exist)
        dh = DataHandler(state)
        await dh.execute()
    except (ConnectionFailedError, ConnectionError) as exception:
        logger.error(exception)
        await app.stop()


@app.agent(performance_topic)
@database_async_client_decorator
async def kafka_reader(
    stream,
    state: State,
) -> None:
    reader = KafkaReader(state)
    async for event in stream.events():
        log_message = f'Event message. Partition – Offset: {event.message.partition} - {event.message.offset}'
        logger.info(msg=log_message)
        try:
            reader.execute(event)
        except (ConnectionFailedError, ConnectionError) as exception:
            logger.error(exception)
            await app.stop()


@app.timer(interval=ACAD_PERF_SEND_DATASPHERE_INTERVAL)
@database_async_client
async def send_to_datasphere(*args, state: State):
    info_message = f'Trying to send data to Datasphere/Monolith. Every {ACAD_PERF_SEND_DATASPHERE_INTERVAL} seconds'
    logger.info(msg=info_message)
    handler = DataHandler(state)
    await handler.execute()


@app.page('/healthz/')
async def state_kafka_reader(self: View, request):
    is_trouble: bool = any([kafka_reader.crashed, kafka_reader.should_stop])

    if is_trouble:
        return self.text(value=StateLivenessProbe.TROUBLE, status=503)
    return self.text(value=StateLivenessProbe.OK, status=200)
